/**
 * BuddyBoss Complete Analytics - Portuguese Version
 * JavaScript for dashboard functionality and Chart.js integration
 */

jQuery(document).ready(function($) {
    'use strict';

    // Initialize the analytics dashboard
    initializeAnalyticsDashboard();

    function initializeAnalyticsDashboard() {
        // Initialize tabs
        initializeTabs();
        
        // Initialize charts
        initializeCharts();
        
        // Initialize date range selector
        initializeDateRangeSelector();
        
        // Initialize refresh functionality
        initializeRefreshButton();
    }

    function initializeTabs() {
        $('.nav-tab').on('click', function(e) {
            e.preventDefault();
            
            var targetTab = $(this).attr('href');
            
            // Remove active class from all tabs and content
            $('.nav-tab').removeClass('nav-tab-active');
            $('.bb-tab-content').removeClass('active');
            
            // Add active class to clicked tab and corresponding content
            $(this).addClass('nav-tab-active');
            $(targetTab).addClass('active');
            
            // Refresh charts in the active tab
            refreshChartsInTab(targetTab);
        });
    }

    function initializeCharts() {
        // Initialize all charts with Portuguese labels
        if (typeof bbAnalyticsData !== 'undefined' && bbAnalyticsData.data) {
            initializeMembersGrowthChart();
            initializeMemberTypesChart();
            initializeActivityTimelineChart();
            initializeActivityTypesChart();
            initializeGroupsGrowthChart();
            initializeGroupsByTypeChart();
            initializeConnectionsGrowthChart();
            initializeSessionStatsChart();
        }
    }

    function initializeMembersGrowthChart() {
        var ctx = document.getElementById('membersGrowthChart');
        if (!ctx || !bbAnalyticsData.data.members) return;

        var chartData = bbAnalyticsData.data.members.growth || [];
        var labels = chartData.map(item => formatDate(item.date));
        var data = chartData.map(item => parseInt(item.count));

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: bbAnalyticsData.labels.memberGrowth || 'Crescimento de Membros',
                    data: data,
                    borderColor: '#0073aa',
                    backgroundColor: 'rgba(0, 115, 170, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: bbAnalyticsData.labels.memberGrowth || 'Crescimento de Membros',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Número de Membros'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Data'
                        }
                    }
                }
            }
        });
    }

    function initializeMemberTypesChart() {
        var ctx = document.getElementById('memberTypesChart');
        if (!ctx) return;

        // Sample data - this would be populated from PHP
        var data = {
            labels: ['Standard', 'Premium', 'Moderador', 'Administrador'],
            datasets: [{
                data: [65, 20, 10, 5],
                backgroundColor: [
                    '#0073aa',
                    '#00a0d2',
                    '#007cba',
                    '#005177'
                ],
                borderWidth: 2,
                borderColor: '#fff'
            }]
        };

        new Chart(ctx, {
            type: 'doughnut',
            data: data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: bbAnalyticsData.labels.memberTypes || 'Tipos de Membros',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true
                        }
                    }
                }
            }
        });
    }

    function initializeActivityTimelineChart() {
        var ctx = document.getElementById('activityTimelineChart');
        if (!ctx || !bbAnalyticsData.data.activity) return;

        var chartData = bbAnalyticsData.data.activity.timeline || [];
        var labels = chartData.map(item => formatDate(item.date));
        var data = chartData.map(item => parseInt(item.count));

        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: bbAnalyticsData.labels.activityTimeline || 'Atividade Diária',
                    data: data,
                    backgroundColor: 'rgba(0, 115, 170, 0.8)',
                    borderColor: '#0073aa',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: bbAnalyticsData.labels.activityTimeline || 'Linha do Tempo da Atividade',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Número de Atividades'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Data'
                        }
                    }
                }
            }
        });
    }

    function initializeActivityTypesChart() {
        var ctx = document.getElementById('activityTypesChart');
        if (!ctx) return;

        // Sample data - this would be populated from PHP
        var data = {
            labels: ['Publicações', 'Comentários', 'Novos Membros', 'Amizades', 'Grupos'],
            datasets: [{
                data: [45, 30, 15, 7, 3],
                backgroundColor: [
                    '#0073aa',
                    '#00a0d2',
                    '#007cba',
                    '#005177',
                    '#003f5c'
                ],
                borderWidth: 2,
                borderColor: '#fff'
            }]
        };

        new Chart(ctx, {
            type: 'pie',
            data: data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: bbAnalyticsData.labels.activityTypes || 'Tipos de Atividade',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true
                        }
                    }
                }
            }
        });
    }

    function initializeGroupsGrowthChart() {
        var ctx = document.getElementById('groupsGrowthChart');
        if (!ctx || !bbAnalyticsData.data.groups) return;

        var chartData = bbAnalyticsData.data.groups.growth || [];
        var labels = chartData.map(item => formatDate(item.date));
        var data = chartData.map(item => parseInt(item.count));

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: bbAnalyticsData.labels.groupGrowth || 'Crescimento de Grupos',
                    data: data,
                    borderColor: '#46b450',
                    backgroundColor: 'rgba(70, 180, 80, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: bbAnalyticsData.labels.groupGrowth || 'Crescimento de Grupos',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Número de Grupos'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Data'
                        }
                    }
                }
            }
        });
    }

    function initializeGroupsByTypeChart() {
        var ctx = document.getElementById('groupsByTypeChart');
        if (!ctx) return;

        // Sample data - this would be populated from PHP
        var data = {
            labels: ['Público', 'Privado', 'Oculto'],
            datasets: [{
                data: [60, 30, 10],
                backgroundColor: [
                    '#46b450',
                    '#ffb900',
                    '#dc3232'
                ],
                borderWidth: 2,
                borderColor: '#fff'
            }]
        };

        new Chart(ctx, {
            type: 'doughnut',
            data: data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: bbAnalyticsData.labels.groupsByType || 'Grupos por Tipo',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true
                        }
                    }
                }
            }
        });
    }

    function initializeConnectionsGrowthChart() {
        var ctx = document.getElementById('connectionsGrowthChart');
        if (!ctx || !bbAnalyticsData.data.network) return;

        var friendshipsData = bbAnalyticsData.data.network.friendships || [];
        var labels = friendshipsData.map(item => formatDate(item.date));
        var data = friendshipsData.map(item => parseInt(item.count));

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: bbAnalyticsData.labels.friendships || 'Amizades',
                    data: data,
                    borderColor: '#f56e28',
                    backgroundColor: 'rgba(245, 110, 40, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Crescimento de Ligações',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Número de Ligações'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Data'
                        }
                    }
                }
            }
        });
    }

    function initializeSessionStatsChart() {
        var ctx = document.getElementById('sessionStatsChart');
        if (!ctx) return;

        // Sample data for session statistics
        var data = {
            labels: ['0-5 min', '5-15 min', '15-30 min', '30-60 min', '60+ min'],
            datasets: [{
                label: 'Duração da Sessão',
                data: [25, 35, 20, 15, 5],
                backgroundColor: 'rgba(156, 39, 176, 0.8)',
                borderColor: '#9c27b0',
                borderWidth: 1
            }]
        };

        new Chart(ctx, {
            type: 'bar',
            data: data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Distribuição da Duração das Sessões',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Percentagem de Utilizadores (%)'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Duração da Sessão'
                        }
                    }
                }
            }
        });
    }

    function initializeDateRangeSelector() {
        $('#bb-date-range').on('change', function() {
            var selectedRange = $(this).val();
            updateDashboardData(selectedRange);
        });
    }

    function initializeRefreshButton() {
        // Add refresh button if it doesn't exist
        if ($('.bb-refresh-button').length === 0) {
            $('.bb-date-range-selector').append(
                '<button type="button" class="button bb-refresh-button" style="margin-left: 10px;">' +
                'Atualizar Dados' +
                '</button>'
            );
        }

        $('.bb-refresh-button').on('click', function() {
            var selectedRange = $('#bb-date-range').val();
            updateDashboardData(selectedRange, true);
        });
    }

    function updateDashboardData(dateRange, forceRefresh = false) {
        // Show loading indicator
        showLoadingIndicator();

        $.post(bbAnalyticsData.ajaxurl, {
            action: 'bb_analytics_update_range',
            date_range: dateRange,
            force_refresh: forceRefresh,
            nonce: bbAnalyticsData.nonce
        })
        .done(function(response) {
            if (response.success) {
                // Update the data and refresh charts
                bbAnalyticsData.data = response.data;
                refreshAllCharts();
                updateSummaryCards(response.data.summary);
            } else {
                showErrorMessage('Erro ao atualizar dados: ' + response.data);
            }
        })
        .fail(function() {
            showErrorMessage('Erro de conexão ao atualizar dados.');
        })
        .always(function() {
            hideLoadingIndicator();
        });
    }

    function refreshAllCharts() {
        // Destroy existing charts and recreate them
        Chart.helpers.each(Chart.instances, function(instance) {
            instance.destroy();
        });
        
        // Reinitialize all charts
        setTimeout(function() {
            initializeCharts();
        }, 100);
    }

    function refreshChartsInTab(tabId) {
        // Refresh specific charts based on the active tab
        setTimeout(function() {
            switch(tabId) {
                case '#members':
                    initializeMembersGrowthChart();
                    initializeMemberTypesChart();
                    break;
                case '#activity':
                    initializeActivityTimelineChart();
                    initializeActivityTypesChart();
                    break;
                case '#groups':
                    initializeGroupsGrowthChart();
                    initializeGroupsByTypeChart();
                    break;
                case '#network':
                    initializeConnectionsGrowthChart();
                    break;
                case '#engagement':
                    initializeSessionStatsChart();
                    break;
            }
        }, 100);
    }

    function updateSummaryCards(summaryData) {
        if (!summaryData) return;

        summaryData.forEach(function(metric, index) {
            var card = $('.bb-summary-card').eq(index);
            if (card.length) {
                card.find('.bb-summary-value').text(formatNumber(metric.value));
                if (metric.change !== undefined) {
                    var changeElement = card.find('.bb-summary-change');
                    var changeClass = metric.change >= 0 ? 'positive' : 'negative';
                    var changeText = (metric.change >= 0 ? '+' : '') + metric.change.toFixed(1) + '%';
                    
                    changeElement
                        .removeClass('positive negative')
                        .addClass(changeClass)
                        .text(changeText);
                }
            }
        });
    }

    function showLoadingIndicator() {
        if ($('.bb-loading-overlay').length === 0) {
            $('body').append(
                '<div class="bb-loading-overlay">' +
                '<div class="bb-loading-spinner">' +
                '<div class="spinner"></div>' +
                '<p>A carregar dados...</p>' +
                '</div>' +
                '</div>'
            );
        }
        $('.bb-loading-overlay').fadeIn(200);
    }

    function hideLoadingIndicator() {
        $('.bb-loading-overlay').fadeOut(200);
    }

    function showErrorMessage(message) {
        // Remove existing error messages
        $('.bb-error-message').remove();
        
        // Add new error message
        $('.bb-analytics-wrap').prepend(
            '<div class="notice notice-error bb-error-message is-dismissible">' +
            '<p>' + message + '</p>' +
            '<button type="button" class="notice-dismiss">' +
            '<span class="screen-reader-text">Dispensar este aviso.</span>' +
            '</button>' +
            '</div>'
        );
        
        // Handle dismiss button
        $('.notice-dismiss').on('click', function() {
            $(this).parent().fadeOut(300);
        });
        
        // Auto-hide after 5 seconds
        setTimeout(function() {
            $('.bb-error-message').fadeOut(300);
        }, 5000);
    }

    // Utility functions
    function formatDate(dateString) {
        var date = new Date(dateString);
        return date.toLocaleDateString('pt-PT', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric'
        });
    }

    function formatNumber(number) {
        return new Intl.NumberFormat('pt-PT').format(number);
    }

    // Handle responsive behavior
    $(window).on('resize', function() {
        // Redraw charts on window resize
        Chart.helpers.each(Chart.instances, function(instance) {
            instance.resize();
        });
    });

    // Initialize tooltips if needed
    $('[data-tooltip]').each(function() {
        $(this).attr('title', $(this).data('tooltip'));
    });

    // Handle print functionality
    window.addEventListener('beforeprint', function() {
        Chart.helpers.each(Chart.instances, function(instance) {
            instance.resize();
        });
    });
});

// Export functions for external use if needed
window.BuddyBossAnalytics = {
    refreshCharts: function() {
        Chart.helpers.each(Chart.instances, function(instance) {
            instance.destroy();
        });
        setTimeout(function() {
            jQuery(document).ready(function() {
                initializeCharts();
            });
        }, 100);
    }
};

